import { PrismaClient } from '@prisma/client';
import bcrypt from 'bcryptjs';

const hashPassword = async (password: string): Promise<string> => {
  return bcrypt.hash(password, 12);
};

const prisma = new PrismaClient();

async function main() {
  console.log('🌱 Seeding database...');

  // Create admin user
  const adminPassword = await hashPassword('admin123');
  const admin = await prisma.user.upsert({
    where: { email: 'admin@ecoscope.ro' },
    update: {},
    create: {
      email: 'admin@ecoscope.ro',
      name: 'Administrator',
      password: adminPassword,
      role: 'ADMIN',
    },
  });

  // Create demo user
  const userPassword = await hashPassword('user123');
  const user = await prisma.user.upsert({
    where: { email: 'demo@ecoscope.ro' },
    update: {},
    create: {
      email: 'demo@ecoscope.ro',
      name: 'Utilizator Demo',
      password: userPassword,
      role: 'USER',
    },
  });

  console.log('✅ Database seeded successfully!');
  console.log('👤 Admin user:', admin.email, '(password: admin123)');
  console.log('👤 Demo user:', user.email, '(password: user123)');
}

main()
  .catch((e) => {
    console.error('❌ Error seeding database:', e);
    process.exit(1);
  })
  .finally(async () => {
    await prisma.$disconnect();
  });
