import { NextRequest, NextResponse } from 'next/server';
import { prisma } from '@/lib/prisma';
import { verifyToken, verifyPassword, hashPassword } from '@/lib/auth';
import { z } from 'zod';

const updatePasswordSchema = z.object({
  currentPassword: z.string().min(1, 'Parola curentă este obligatorie'),
  newPassword: z.string().min(6, 'Parola nouă trebuie să aibă cel puțin 6 caractere'),
});

export async function PUT(request: NextRequest) {
  try {
    const token = request.headers.get('authorization')?.replace('Bearer ', '');
    
    if (!token) {
      return NextResponse.json(
        { error: 'Token de autentificare lipsă' },
        { status: 401 }
      );
    }

    const payload = verifyToken(token);
    if (!payload) {
      return NextResponse.json(
        { error: 'Token invalid' },
        { status: 401 }
      );
    }

    const body = await request.json();
    const { currentPassword, newPassword } = updatePasswordSchema.parse(body);

    // Găsește utilizatorul
    const user = await prisma.user.findUnique({
      where: { id: payload.userId },
    });

    if (!user) {
      return NextResponse.json(
        { error: 'Utilizatorul nu a fost găsit' },
        { status: 404 }
      );
    }

    // Verifică parola curentă
    const isCurrentPasswordValid = await verifyPassword(currentPassword, user.password);
    if (!isCurrentPasswordValid) {
      return NextResponse.json(
        { error: 'Parola curentă este incorectă' },
        { status: 400 }
      );
    }

    // Hash parola nouă
    const hashedNewPassword = await hashPassword(newPassword);

    // Actualizează parola
    await prisma.user.update({
      where: { id: payload.userId },
      data: { password: hashedNewPassword },
    });

    return NextResponse.json({
      message: 'Parola a fost actualizată cu succes',
    });
  } catch (error) {
    if (error instanceof z.ZodError) {
      return NextResponse.json(
        { error: 'Date invalide', details: error.issues },
        { status: 400 }
      );
    }

    console.error('Error updating password:', error);
    return NextResponse.json(
      { error: 'A apărut o eroare la actualizarea parolei' },
      { status: 500 }
    );
  }
}
