import { NextRequest, NextResponse } from 'next/server';
import { prisma } from '@/lib/prisma';
import { verifyToken } from '@/lib/auth';
import { z } from 'zod';

const updateProfileSchema = z.object({
  name: z.string().min(2, 'Numele trebuie să aibă cel puțin 2 caractere'),
  email: z.string().email('Adresa de email nu este validă'),
});

export async function PUT(request: NextRequest) {
  try {
    const token = request.headers.get('authorization')?.replace('Bearer ', '');
    
    if (!token) {
      return NextResponse.json(
        { error: 'Token de autentificare lipsă' },
        { status: 401 }
      );
    }

    const payload = verifyToken(token);
    if (!payload) {
      return NextResponse.json(
        { error: 'Token invalid' },
        { status: 401 }
      );
    }

    const body = await request.json();
    const { name, email } = updateProfileSchema.parse(body);

    // Verifică dacă email-ul este deja folosit de alt utilizator
    const existingUser = await prisma.user.findFirst({
      where: {
        email,
        id: { not: payload.userId },
      },
    });

    if (existingUser) {
      return NextResponse.json(
        { error: 'Această adresă de email este deja folosită' },
        { status: 400 }
      );
    }

    // Actualizează utilizatorul
    const updatedUser = await prisma.user.update({
      where: { id: payload.userId },
      data: { name, email },
      select: {
        id: true,
        name: true,
        email: true,
        role: true,
        createdAt: true,
        updatedAt: true,
      },
    });

    return NextResponse.json({
      user: updatedUser,
      token, // Returnează același token
    });
  } catch (error) {
    if (error instanceof z.ZodError) {
      return NextResponse.json(
        { error: 'Date invalide', details: error.issues },
        { status: 400 }
      );
    }

    console.error('Error updating profile:', error);
    return NextResponse.json(
      { error: 'A apărut o eroare la actualizarea profilului' },
      { status: 500 }
    );
  }
}
