"use client";

import { useState } from "react";
import Link from "next/link";

export default function ContactPage() {
  const [formData, setFormData] = useState({
    name: "",
    email: "",
    subject: "",
    message: "",
  });
  const [isSubmitting, setIsSubmitting] = useState(false);
  const [submitMessage, setSubmitMessage] = useState<string | null>(null);

  const handleSubmit = async (e: React.FormEvent) => {
    e.preventDefault();
    setIsSubmitting(true);
    setSubmitMessage(null);

    // Simulare trimitere formular (în producție ar fi o apelare API)
    setTimeout(() => {
      setIsSubmitting(false);
      setSubmitMessage("Mesajul a fost trimis cu succes! Vă vom contacta în curând.");
      setFormData({ name: "", email: "", subject: "", message: "" });
    }, 2000);
  };

  const handleChange = (e: React.ChangeEvent<HTMLInputElement | HTMLTextAreaElement | HTMLSelectElement>) => {
    setFormData({
      ...formData,
      [e.target.name]: e.target.value,
    });
  };

  return (
    <main className="mx-auto max-w-4xl p-6">
      <div className="mb-8">
        <Link href="/" className="inline-flex items-center text-emerald-700 hover:text-emerald-800 mb-4">
          <svg className="w-4 h-4 mr-2" fill="none" stroke="currentColor" viewBox="0 0 24 24">
            <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M10 19l-7-7m0 0l7-7m-7 7h18" />
          </svg>
          Înapoi la pagina principală
        </Link>
        <h1 className="text-3xl font-bold text-emerald-800 mb-4">Contact</h1>
        <p className="text-lg text-gray-600">
          Aveți întrebări despre EcoScope sau aveți nevoie de asistență? Suntem aici să vă ajutăm.
        </p>
      </div>

      <div className="grid grid-cols-1 lg:grid-cols-2 gap-8">
        {/* Formular de contact */}
        <div className="bg-white rounded-lg border border-gray-200 p-6">
          <h2 className="text-xl font-semibold text-emerald-800 mb-4">Trimiteți-ne un mesaj</h2>
          <form onSubmit={handleSubmit} className="space-y-4">
            <div>
              <label htmlFor="name" className="block text-sm font-medium text-gray-700 mb-1">
                Nume complet *
              </label>
              <input
                type="text"
                id="name"
                name="name"
                required
                value={formData.name}
                onChange={handleChange}
                className="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-emerald-500 focus:border-emerald-500"
                placeholder="Introduceți numele complet"
              />
            </div>

            <div>
              <label htmlFor="email" className="block text-sm font-medium text-gray-700 mb-1">
                Adresă email *
              </label>
              <input
                type="email"
                id="email"
                name="email"
                required
                value={formData.email}
                onChange={handleChange}
                className="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-emerald-500 focus:border-emerald-500"
                placeholder="exemplu@email.com"
              />
            </div>

            <div>
              <label htmlFor="subject" className="block text-sm font-medium text-gray-700 mb-1">
                Subiect *
              </label>
              <select
                id="subject"
                name="subject"
                required
                value={formData.subject}
                onChange={handleChange}
                className="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-emerald-500 focus:border-emerald-500"
              >
                <option value="">Selectați un subiect</option>
                <option value="support">Suport tehnic</option>
                <option value="methodology">Întrebări metodologie</option>
                <option value="pricing">Prețuri și licențe</option>
                <option value="partnership">Parteneriat</option>
                <option value="feedback">Feedback</option>
                <option value="other">Altele</option>
              </select>
            </div>

            <div>
              <label htmlFor="message" className="block text-sm font-medium text-gray-700 mb-1">
                Mesaj *
              </label>
              <textarea
                id="message"
                name="message"
                required
                rows={5}
                value={formData.message}
                onChange={handleChange}
                className="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-emerald-500 focus:border-emerald-500"
                placeholder="Descrieți întrebarea sau problema dumneavoastră..."
              />
            </div>

            <button
              type="submit"
              disabled={isSubmitting}
              className="w-full bg-emerald-600 text-white py-2 px-4 rounded-md hover:bg-emerald-700 focus:outline-none focus:ring-2 focus:ring-emerald-500 focus:ring-offset-2 disabled:opacity-50 disabled:cursor-not-allowed"
            >
              {isSubmitting ? "Se trimite..." : "Trimite mesajul"}
            </button>

            {submitMessage && (
              <div className="p-3 bg-green-50 border border-green-200 rounded-md">
                <p className="text-green-700 text-sm">{submitMessage}</p>
              </div>
            )}
          </form>
        </div>

        {/* Informații de contact */}
        <div className="space-y-6">
          <div className="bg-white rounded-lg border border-gray-200 p-6">
            <h2 className="text-xl font-semibold text-emerald-800 mb-4">Informații de contact</h2>
            <div className="space-y-4">
              <div className="flex items-start">
                <svg className="w-5 h-5 text-emerald-600 mr-3 mt-0.5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                  <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M3 8l7.89 4.26a2 2 0 002.22 0L21 8M5 19h14a2 2 0 002-2V7a2 2 0 00-2-2H5a2 2 0 00-2 2v10a2 2 0 002 2z" />
                </svg>
                <div>
                  <h3 className="font-medium text-gray-800">Email</h3>
                  <p className="text-sm text-gray-600">contact@ecoscope.ro</p>
                  <p className="text-sm text-gray-600">support@ecoscope.ro</p>
                </div>
              </div>

              <div className="flex items-start">
                <svg className="w-5 h-5 text-emerald-600 mr-3 mt-0.5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                  <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M3 5a2 2 0 012-2h3.28a1 1 0 01.948.684l1.498 4.493a1 1 0 01-.502 1.21l-2.257 1.13a11.042 11.042 0 005.516 5.516l1.13-2.257a1 1 0 011.21-.502l4.493 1.498a1 1 0 01.684.949V19a2 2 0 01-2 2h-1C9.716 21 3 14.284 3 6V5z" />
                </svg>
                <div>
                  <h3 className="font-medium text-gray-800">Telefon</h3>
                  <p className="text-sm text-gray-600">+40 21 123 4567</p>
                  <p className="text-sm text-gray-500">Luni - Vineri: 9:00 - 18:00</p>
                </div>
              </div>

              <div className="flex items-start">
                <svg className="w-5 h-5 text-emerald-600 mr-3 mt-0.5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                  <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M17.657 16.657L13.414 20.9a1.998 1.998 0 01-2.827 0l-4.244-4.243a8 8 0 1111.314 0z" />
                  <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M15 11a3 3 0 11-6 0 3 3 0 016 0z" />
                </svg>
                <div>
                  <h3 className="font-medium text-gray-800">Adresă</h3>
                  <p className="text-sm text-gray-600">
                    Strada Ecologiei, Nr. 123<br />
                    Sector 1, București 010001<br />
                    România
                  </p>
                </div>
              </div>
            </div>
          </div>

          <div className="bg-white rounded-lg border border-gray-200 p-6">
            <h2 className="text-xl font-semibold text-emerald-800 mb-4">Suport tehnic</h2>
            <div className="space-y-3">
              <div className="flex items-start">
                <svg className="w-4 h-4 text-emerald-600 mr-2 mt-0.5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                  <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M9 12l2 2 4-4m6 2a9 9 0 11-18 0 9 9 0 0118 0z" />
                </svg>
                <p className="text-sm text-gray-600">Răspuns în 24 de ore pentru întrebări generale</p>
              </div>
              <div className="flex items-start">
                <svg className="w-4 h-4 text-emerald-600 mr-2 mt-0.5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                  <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M9 12l2 2 4-4m6 2a9 9 0 11-18 0 9 9 0 0118 0z" />
                </svg>
                <p className="text-sm text-gray-600">Suport tehnic specializat pentru probleme complexe</p>
              </div>
              <div className="flex items-start">
                <svg className="w-4 h-4 text-emerald-600 mr-2 mt-0.5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                  <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M9 12l2 2 4-4m6 2a9 9 0 11-18 0 9 9 0 0118 0z" />
                </svg>
                <p className="text-sm text-gray-600">Training și consultanță pentru implementare</p>
              </div>
            </div>
          </div>

          <div className="bg-white rounded-lg border border-gray-200 p-6">
            <h2 className="text-xl font-semibold text-emerald-800 mb-4">FAQ</h2>
            <div className="space-y-3">
              <div>
                <h3 className="font-medium text-gray-800 text-sm">Cât costă utilizarea EcoScope?</h3>
                <p className="text-sm text-gray-600">Versiunea demo este gratuită. Pentru versiunea completă, contactați-ne pentru un preț personalizat.</p>
              </div>
              <div>
                <h3 className="font-medium text-gray-800 text-sm">Pot exporta rapoartele?</h3>
                <p className="text-sm text-gray-600">Da, toate rapoartele pot fi exportate în format PDF pentru audit și verificare.</p>
              </div>
              <div>
                <h3 className="font-medium text-gray-800 text-sm">Sunt datele mele sigure?</h3>
                <p className="text-sm text-gray-600">Da, toate datele sunt criptate și stocate în conformitate cu GDPR.</p>
              </div>
            </div>
          </div>
        </div>
      </div>
    </main>
  );
}
