export type FuelType = 'diesel' | 'gasoline' | 'lpg' | 'natural_gas' | 'biomass';

export type Tillage = 'plough' | 'min_till' | 'no_till';

// 1. Date de bază despre organizație/firma
export interface Organization {
  name: string; // Denumire
  location: string; // Locație
  activityDomain: string; // Domeniu de activitate (agricultură, procesare, logistică etc.)
  cultivatedAreaHa: number; // Suprafață agricolă cultivată / operată
  cropTypes: string; // Tipuri de culturi / producție agricolă
  animalCount?: number; // Număr de animale (dacă activitatea implică zootehnie)
  infrastructure: string; // Infrastructură fizică utilizată (ferme, silozuri, utilaje mari etc.)
}

// 2. Consum de energie (electricitate, combustibili)
export interface EnergyInputs {
  electricityKWh: number; // Facturi la curent electric pentru ferme, clădiri, spații de birouri etc.
  renewableSharePct?: number; // Surse de energie regenerabilă și proporția acestora (0..100)
  fuels: Array<{ type: FuelType; amount: number; unit: 'L' | 'kg' }>; // Tipuri de combustibili folosiți (motorină, benzină, GPL, gaz, biomasa etc.), volume/consum anual
}


export interface AgriInputs {
  // 3. Utilizarea inputurilor agricole
  // o Cantități și tipuri de îngrășăminte folosite (nitrogen, fosfați, potasiu etc.)
  fertilizerN_Kg: number; // Cantitate Azot (N) - kg/an
  fertilizerN_Type?: string; // Tip îngrășământ Azot
  fertilizerP2O5_Kg?: number; // Cantitate Fosfor (P2O5) - kg/an
  fertilizerP_Type?: string; // Tip îngrășământ Fosfor
  fertilizerK2O_Kg?: number; // Cantitate Potasiu (K2O) - kg/an
  fertilizerK_Type?: string; // Tip îngrășământ Potasiu
  // o Pesticide, erbicide, fungicide – tipuri, doze, frecvență
  pesticideType?: string; // Tip pesticid (ex: erbicid, fungicid)
  pesticideDosePerHa?: number; // Doză per hectar (kg sau L)
  pesticideApplicationsPerSeason?: number; // Frecvență (aplicări/sezon)
  // o Semănături, semințe, material săditor
  seedType?: string; // Tip semințe/material săditor
  // o Proporția inputurilor importate vs locale (transport, emisie asociată)
  importSharePct?: number; // Proporția inputurilor importate (%) - 0..100
}

export interface LandManagement {
  // 4. Managementul terenului și solului
  // o Tipul de sol, utilizare precedentă (rotatie, monoculturi, teren necultivat)
  soilType?: string; // Tipul de sol
  previousUse?: string; // Utilizare precedentă (rotație, monoculturi, teren necultivat)
  // o Practici agricole: irigare, drenaj, arături / lucrare minimă a solului, agroforestry, acoperire vegetală / culturi intermediare
  irrigationM3?: number; // Irigare (m³/an)
  drainage?: boolean; // Drenaj
  tillage: Tillage; // Arături / lucrare minimă a solului
  agroforestry?: boolean; // Agroforestry
  coverCropsHa?: number; // Acoperire vegetală / culturi intermediare (ha)
  // o Eroziune, degradare, fertilitatea solului (analize de sol) dacă sunt disponibile
  erosionDegradationNotes?: string; // Eroziune, degradare (note)
  soilFertilityNotes?: string; // Fertilitatea solului (analize de sol)
}


// 5. Transporte și lanț de aprovizionare
export interface TransportSupplyChain {
  internalTransport: {
    distances: number; // Transporturi interne: distanțe
    vehicleType: string; // tip de vehicule
    frequency: string; // frecvență
  };
  inputTransport: {
    fuelTransport: string; // Transporturile de inputuri (combustibili)
    seedTransport: string; // semințe
    chemicalInputsTransport: string; // inputuri chimice
  };
  exportCollection: {
    productionExport: string; // exporturile/colectările producției
  };
  storageProcessing: {
    storage: string; // Depozitare
    processing: string; // procesare
    packaging: string; // ambalare
    equipment: string; // utilaje
    energyConsumed: number; // energie consumată
  };
}

// 6. Emisii directe și indirecte
export interface DirectIndirectEmissions {
  directEmissions: {
    onSiteEngines: string; // Emisii directe la fața locului (ex: motoarele utilajelor, generatoare, echipamente)
    onSiteGenerators: string; // Generatoare
    onSiteEquipment: string; // Echipamente
  };
  indirectEmissions: {
    importedElectricity: string; // Emisii indirecte prin electricitate importată
    heating: string; // încălzire etc.
  };
  fertilizerPesticideEmissions: {
    fertilizerEmissions: string; // Emisii din fertilizanți
    pesticideEmissions: string; // pesticid
    lcaModels: string; // dacă platforma permite LCA și modele de emisie globale
  };
}

// 7. Factori de emisie (emission factors)
export interface EmissionFactors {
  electricity_kgCO2e_per_kWh: number; // Factorii de emisie pentru electricitate
  fuel_kgCO2e_per_L: Record<FuelType, number>; // pentru fiecare tip de combustibil
  n2o_kgCO2e_per_kgN: number; // pentru inputuri chimice (IPCC Tier 1 conversion)
  irrigation_kgCO2e_per_m3?: number; // pentru irigare
  soil_sequestration_tCO2e_per_ha?: number; // negative for sequestration
  // Factorii de emisie ideal localizați (factor românesc / UE) sau generici recunoscuți (ex. IPCC, baze de date EU)
}

// Extended emission factors for new categories
export interface ExtendedEmissionFactors extends EmissionFactors {
  transport_kgCO2e_per_km?: number; // pentru transport
  storage_kgCO2e_per_kWh?: number; // pentru depozitare/procesare
  packaging_kgCO2e_per_tonne?: number; // pentru ambalare
  pesticide_kgCO2e_per_kg?: number; // pentru pesticide
  seed_kgCO2e_per_kg?: number; // pentru semințe
  waste_kgCO2e_per_tonne?: number; // pentru deșeuri agricole
}

// 8. Date despre producție / output
export interface ProductionOutput {
  quantities: {
    tonnage: number; // Cantități produse (tonaje)
    volume: number; // volume
  };
  productionCalendar: {
    annual: string; // Calendarul producției (anual)
    seasons: string; // sezoane etc.
  };
  productionType: {
    green: string; // Tipul de producție (verdele)
    processing: string; // procesare
    rawMaterial: string; // materie primă etc.
  };
  finishingDistribution: {
    finishing: string; // Finisare
    packaging: string; // ambalaj
    distribution: string; // distribuție
  };
}

// 9. Date financiare relevante
export interface FinancialData {
  inputCosts: {
    energyCosts: number; // Costuri legate de energie
    fuelCosts: number; // combustibili
    fertilizerCosts: number; // fertilizanți etc.
  };
  salesPrices: {
    sellingPrices: number; // Prețuri de vânzare
    margins: number; // marje (pentru estimări de cost-beneficiu sau pentru scenarii de decarbonizare)
  };
  transitionCosts: {
    technologyInvestments: number; // Costuri de tranziție (investiții în tehnologii)
    greenEquipment: number; // echipamente verzi
    modernization: number; // modernizare
  };
}

// 10. Alte date auxiliare
export interface AuxiliaryData {
  irrigationClimate: {
    irrigationCalendar: string; // Calendar/regimul de irigare
    localClimate: string; // climă locală (temperaturi, precipitații) – pentru calcule legate de irigare și emisii din sol
    temperatureData: string; // Temperaturi
    precipitationData: string; // Precipitații
  };
  wasteManagement: {
    agriculturalWaste: string; // Practici pentru managementul deșeurilor agricole
    residues: string; // reziduuri
  };
  personnelData: {
    workingHours: number; // Date despre personal (ore de muncă)
    personalTransport: string; // transport personal dacă există implicare în emisie de carbon – Scope 3
  };
}

export interface Scenario {
  // 1. Date de bază despre organizație/firma
  org: Organization;
  // 2. Consum de energie (electricitate, combustibili)
  energy: EnergyInputs;
  // 3. Utilizarea inputurilor agricole
  agri: AgriInputs;
  // 4. Managementul terenului și solului
  land: LandManagement;
  // 5. Transporte și lanț de aprovizionare
  transport: TransportSupplyChain;
  // 6. Emisii directe și indirecte
  emissions: DirectIndirectEmissions;
  // 7. Factori de emisie (emission factors)
  factors: EmissionFactors;
  // 8. Date despre producție / output
  production: ProductionOutput;
  // 9. Date financiare relevante
  financial: FinancialData;
  // 10. Alte date auxiliare
  auxiliary: AuxiliaryData;
  // Preț per tonă CO2 pentru calculul certificatelor
  pricePerTonneEUR: number;
}

export interface EmissionsBreakdown {
  // Amprenta de carbon (Carbon footprint)
  scope1: number; // tCO2e - Emisii directe (combustibili, fertilizanți, irigare)
  scope2: number; // tCO2e - Emisii indirecte (electricitate)
  scope3: number; // tCO2e - Emisii indirecte (transport, inputuri importate)
  total: number; // tCO2e - Emisii totale de gaze cu efect de seră (GHG)
  
  // Identificarea "hotspoturilor" de emisii
  hotspots: {
    energy: number; // tCO2e - energia (electricitate + combustibili)
    inputs: number; // tCO2e - inputuri (fertilizanți, pesticide, semințe)
    transport: number; // tCO2e - transport și lanț de aprovizionare
    irrigation: number; // tCO2e - irigare
    processing: number; // tCO2e - procesare, depozitare, ambalare
    waste: number; // tCO2e - deșeuri agricole
    soil: number; // tCO2e - sechestrarea carbonului din sol (negativ)
  };
  
  // Detalii pentru fiecare categorie
  details: Record<string, number>; // tCO2e per category
}

export function computeEmissions(s: Scenario): EmissionsBreakdown {
  const re = (s.energy.renewableSharePct ?? 0) / 100;

  // Scope 2: electricity (emisii indirecte)
  const scope2 = (s.energy.electricityKWh * (1 - re) * s.factors.electricity_kgCO2e_per_kWh) / 1000;

  // Scope 1: fuels (emisii directe - tillage affects fuel consumption)
  const fuelsKgCO2e = s.energy.fuels.reduce((sum, f) => {
    const ef = s.factors.fuel_kgCO2e_per_L[f.type] ?? 0;
    const amountKg = f.unit === 'kg' ? f.amount : f.amount * ef;
    const kgCO2e = f.unit === 'kg' ? amountKg : f.amount * ef;
    return sum + kgCO2e;
  }, 0);
  const scope1Fuels = fuelsKgCO2e / 1000;

  // Scope 1: N2O from nitrogen fertilizer (emisii directe din inputuri)
  const scope1N = (s.agri.fertilizerN_Kg * s.factors.n2o_kgCO2e_per_kgN) / 1000;

  // Scope 1: irrigation emissions (emisii directe din irigare)
  const irrigationEmissions = s.land.irrigationM3 
    ? (s.land.irrigationM3 * (s.factors.irrigation_kgCO2e_per_m3 ?? 0)) / 1000
    : 0;

  // Soil sequestration from cover crops (sechestrarea carbonului din sol)
  const sequestrationT = (s.land.coverCropsHa ?? 0) * (s.factors.soil_sequestration_tCO2e_per_ha ?? 0);

  // Enhanced Scope 3: transport și inputuri importate (emisii indirecte)
  // Transporturi interne - calcul mai detaliat
  const internalTransportEmissions = s.transport.internalTransport.distances * 0.15; // factor îmbunătățit
  
  // Transporturi de inputuri - estimare bazată pe distanțe și tipuri
  const inputTransportEmissions = s.transport.internalTransport.distances * 0.05; // transport inputuri
  
  // Depozitare, procesare, ambalare - bazat pe energia consumată
  const storageProcessingEmissions = s.transport.storageProcessing.energyConsumed * 0.27 / 1000; // folosind factor electricitate
  
  // Export/colectare producție - estimare
  const exportCollectionEmissions = s.transport.internalTransport.distances * 0.08;
  
  // Emisii din pesticide (dacă sunt specificate)
  const pesticideEmissions = s.agri.pesticideDosePerHa && s.agri.pesticideApplicationsPerSeason
    ? (s.agri.pesticideDosePerHa * s.agri.pesticideApplicationsPerSeason * s.org.cultivatedAreaHa * 0.5) / 1000 // factor estimat
    : 0;
  
  // Emisii din semințe/material săditor
  const seedEmissions = s.org.cultivatedAreaHa * 0.1; // factor estimat per hectar
  
  // Emisii din deșeuri agricole (Scope 3)
  const wasteEmissions = s.org.cultivatedAreaHa * 0.2; // factor estimat per hectar
  
  // Transport personal (Scope 3)
  const personalTransportEmissions = s.auxiliary.personnelData.workingHours * 0.001; // factor estimat per oră
  
  // Importuri - calcul îmbunătățit
  const importEmissions = (s.agri.importSharePct ?? 0) / 100 * (scope1N + irrigationEmissions + pesticideEmissions + seedEmissions) * 0.15;
  
  const transportEmissions = internalTransportEmissions + inputTransportEmissions + exportCollectionEmissions;
  const processingEmissions = storageProcessingEmissions;
  const scope3 = transportEmissions + importEmissions + processingEmissions + wasteEmissions + personalTransportEmissions;

  const scope1 = scope1Fuels + scope1N + irrigationEmissions + pesticideEmissions + seedEmissions + Math.min(0, sequestrationT);
  const total = scope1 + scope2 + scope3;

  // Identificarea "hotspoturilor" de emisii - îmbunătățită
  const hotspots = {
    energy: scope1Fuels + scope2, // energia (electricitate + combustibili)
    inputs: scope1N + pesticideEmissions + seedEmissions, // inputuri (fertilizanți, pesticide, semințe)
    transport: transportEmissions + importEmissions, // transport și lanț de aprovizionare
    irrigation: irrigationEmissions, // irigare
    processing: processingEmissions, // procesare, depozitare, ambalare
    waste: wasteEmissions, // deșeuri agricole
    soil: Math.min(0, sequestrationT), // sechestrarea carbonului din sol (negativ)
  };

  return {
    scope1,
    scope2,
    scope3,
    total,
    hotspots,
    details: {
      fuels: scope1Fuels,
      fertilizerN2O: scope1N,
      irrigation: irrigationEmissions,
      pesticideEmissions: pesticideEmissions,
      seedEmissions: seedEmissions,
      soilSequestration: Math.min(0, sequestrationT),
      electricity: scope2,
      internalTransport: internalTransportEmissions,
      inputTransport: inputTransportEmissions,
      exportCollection: exportCollectionEmissions,
      storageProcessing: storageProcessingEmissions,
      importEmissions: importEmissions,
      wasteEmissions: wasteEmissions,
      personalTransport: personalTransportEmissions,
    },
  };
}

export interface ComparisonResult {
  baseline: EmissionsBreakdown;
  project: EmissionsBreakdown;
  delta: number; // tCO2e reduced (positive means reduction)
  certificates: number; // tCO2e
  revenueEUR: number;
}

export function compareScenarios(baseline: Scenario, project: Scenario): ComparisonResult {
  const b = computeEmissions(baseline);
  const p = computeEmissions(project);
  const delta = Math.max(0, b.total - p.total);
  const certificates = delta; // 1 credit per tCO2e
  const revenueEUR = certificates * project.pricePerTonneEUR;
  return { baseline: b, project: p, delta, certificates, revenueEUR };
}

// Default emission factors for Romania/EU averages (editable in UI)
export const defaultFactors: EmissionFactors = {
  electricity_kgCO2e_per_kWh: 0.27,
  fuel_kgCO2e_per_L: {
    diesel: 2.68,
    gasoline: 2.31,
    lpg: 1.66,
    natural_gas: 2.05,
    biomass: 0,
  },
  n2o_kgCO2e_per_kgN: 5.96,
  irrigation_kgCO2e_per_m3: 0.15, // demo placeholder
  soil_sequestration_tCO2e_per_ha: -0.3,
};

// Extended emission factors for new categories
export const extendedFactors: ExtendedEmissionFactors = {
  ...defaultFactors,
  transport_kgCO2e_per_km: 0.15, // pentru transport
  storage_kgCO2e_per_kWh: 0.27, // pentru depozitare/procesare (folosește factor electricitate)
  packaging_kgCO2e_per_tonne: 0.5, // pentru ambalare
  pesticide_kgCO2e_per_kg: 0.5, // pentru pesticide
  seed_kgCO2e_per_kg: 0.1, // pentru semințe
  waste_kgCO2e_per_tonne: 0.2, // pentru deșeuri agricole
};


